<?php
require_once '../config.php';

if(!isLoggedIn()) {
    echo '<div class="text-center p-8"><p class="text-red-600">Please login first</p></div>';
    exit();
}

if(!isActiveUser()) {
    echo '<div class="text-center p-8"><p class="text-red-600">Please activate your account first</p></div>';
    exit();
}

$product_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

if(!$product_id) {
    echo '<div class="text-center p-8"><p class="text-red-600">Product not found</p></div>';
    exit();
}

// Get product details
$product_query = "
    SELECT p.*, c.name as category_name 
    FROM products p 
    LEFT JOIN categories c ON p.category_id = c.id 
    WHERE p.id = ? AND p.status = 'active'
";
$stmt = $conn->prepare($product_query);
$stmt->bind_param("i", $product_id);
$stmt->execute();
$product_result = $stmt->get_result();

if($product_result->num_rows == 0) {
    echo '<div class="text-center p-8"><p class="text-red-600">Product not found</p></div>';
    exit();
}

$product = $product_result->fetch_assoc();

// Decode additional images
$additional_images = !empty($product['additional_images']) ? json_decode($product['additional_images'], true) : [];
$all_images = array_merge([$product['image_url']], $additional_images);

// Calculate discount
$discount = 0;
if($product['original_price'] > 0 && $product['sale_price'] > 0) {
    $discount = round((($product['original_price'] - $product['sale_price']) / $product['original_price']) * 100);
}

// Get user balance
$user_id = $_SESSION['user_id'];
$user_balance = getUserBalance($user_id);
?>

<div class="grid grid-cols-1 md:grid-cols-2 gap-8">
    <!-- Product Images -->
    <div>
        <!-- Main Image -->
        <div class="rounded-lg overflow-hidden mb-4">
            <img src="<?php echo $all_images[0]; ?>" 
                 alt="<?php echo htmlspecialchars($product['name']); ?>"
                 class="w-full h-64 object-cover">
        </div>
        
        <!-- Thumbnails -->
        <?php if(count($all_images) > 1): ?>
        <div class="grid grid-cols-4 gap-2">
            <?php foreach($all_images as $index => $image): ?>
            <div class="border-2 border-transparent hover:border-blue-500 rounded overflow-hidden cursor-pointer">
                <img src="<?php echo $image; ?>" 
                     alt="Thumbnail <?php echo $index + 1; ?>"
                     class="w-full h-16 object-cover">
            </div>
            <?php endforeach; ?>
        </div>
        <?php endif; ?>
    </div>
    
    <!-- Product Info -->
    <div>
        <!-- Category -->
        <div class="mb-3">
            <span class="inline-block px-2 py-1 bg-blue-100 text-blue-700 text-xs font-medium rounded">
                <?php echo $product['category_name'] ?: 'Uncategorized'; ?>
            </span>
        </div>
        
        <!-- Product Name -->
        <h2 class="text-2xl font-bold text-gray-800 mb-3">
            <?php echo htmlspecialchars($product['name']); ?>
        </h2>
        
        <!-- Price -->
        <div class="mb-4">
            <?php if($product['original_price'] > 0): ?>
                <div class="flex items-center mb-2">
                    <span class="text-gray-400 line-through text-lg">
                        ৳<?php echo number_format($product['original_price'], 2); ?>
                    </span>
                    <?php if($discount > 0): ?>
                        <span class="ml-3 bg-red-100 text-red-700 px-2 py-1 rounded text-sm font-bold">
                            -<?php echo $discount; ?>%
                        </span>
                    <?php endif; ?>
                </div>
            <?php endif; ?>
            <div class="text-3xl font-bold text-gray-800">
                ৳<?php echo number_format($product['sale_price'], 2); ?>
            </div>
        </div>
        
        <!-- Description -->
        <div class="mb-6">
            <p class="text-gray-600 text-sm">
                <?php echo substr(strip_tags($product['description']), 0, 200); ?>...
            </p>
        </div>
        
        <!-- User Balance -->
        <div class="mb-6 p-4 bg-gray-50 rounded-lg">
            <div class="flex justify-between items-center">
                <span class="text-gray-600">Your Balance:</span>
                <span class="text-xl font-bold <?php echo $user_balance >= $product['sale_price'] ? 'text-green-600' : 'text-red-600'; ?>">
                    ৳<?php echo number_format($user_balance, 2); ?>
                </span>
            </div>
            <?php if($user_balance < $product['sale_price']): ?>
                <div class="mt-2 text-red-600 text-sm">
                    <i class="fas fa-exclamation-triangle mr-1"></i>
                    Need ৳<?php echo number_format($product['sale_price'] - $user_balance, 2); ?> more
                </div>
            <?php endif; ?>
        </div>
        
        <!-- Actions -->
        <div class="space-y-3">
            <a href="product-details.php?id=<?php echo $product['id']; ?>" 
               class="block w-full text-center bg-gradient-to-r from-blue-600 to-indigo-600 text-white font-bold py-3 px-4 rounded-lg hover:from-blue-700 hover:to-indigo-700 transition duration-300">
                <i class="fas fa-eye mr-2"></i> View Full Details
            </a>
            
            <button onclick="addToCart(<?php echo $product['id']; ?>); closeQuickView();" 
                    class="w-full bg-gradient-to-r from-green-500 to-emerald-600 text-white font-bold py-3 px-4 rounded-lg hover:from-green-600 hover:to-emerald-700 transition duration-300">
                <i class="fas fa-cart-plus mr-2"></i> Add to Cart
            </button>
            
            <?php if($user_balance < $product['sale_price']): ?>
                <a href="deposit.php" 
                   class="block w-full text-center bg-gradient-to-r from-yellow-500 to-orange-600 text-white font-bold py-3 px-4 rounded-lg hover:from-yellow-600 hover:to-orange-700 transition duration-300">
                    <i class="fas fa-plus-circle mr-2"></i> Deposit Now
                </a>
            <?php endif; ?>
        </div>
    </div>
</div>

<script>
function addToCart(productId) {
    fetch('ajax/add-to-cart.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: 'product_id=' + productId
    })
    .then(response => response.json())
    .then(data => {
        if(data.success) {
            showNotification('Product added to cart!', 'success');
        } else {
            showNotification(data.message || 'Error adding to cart', 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showNotification('Network error', 'error');
    });
}

function closeQuickView() {
    document.getElementById('quickViewModal').classList.add('hidden');
    document.getElementById('quickViewModal').classList.remove('flex');
}

function showNotification(message, type) {
    // Remove existing notification
    const existing = document.querySelector('.notification-toast');
    if(existing) existing.remove();
    
    // Create notification
    const notification = document.createElement('div');
    notification.className = `notification-toast fixed top-4 right-4 px-6 py-3 rounded-lg shadow-xl text-white font-bold z-50 animate__animated animate__fadeInRight ${type === 'success' ? 'bg-green-500' : 'bg-red-500'}`;
    notification.innerHTML = `
        <div class="flex items-center">
            <i class="fas ${type === 'success' ? 'fa-check-circle' : 'fa-exclamation-circle'} mr-3 text-xl"></i>
            <span>${message}</span>
        </div>
    `;
    document.body.appendChild(notification);
    
    // Remove after 3 seconds
    setTimeout(() => {
        notification.classList.remove('animate__fadeInRight');
        notification.classList.add('animate__fadeOutRight');
        setTimeout(() => notification.remove(), 300);
    }, 3000);
}
</script>